/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/file.c,v 2.192.1.4 1992/11/04 04:47:29 william Exp $";
#endif

#include <sys/types.h>
#include <sys/file.h>
#include <time.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "patchlevel.h"
#include "types.h"

#include "align.e"
#include "arc.e"
#include "attr.e"
#include "auxtext.e"
#include "box.e"
#include "button.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "eps.e"
#include "font.e"
#include "grid.e"
#include "group.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "oval.e"
#include "pattern.e"
#include "poly.e"
#include "polygon.e"
#include "prtgif.e"
#include "ps.e"
#include "raster.e"
#include "rcbox.e"
#include "rect.e"
#include "ruler.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "stk.e"
#include "text.e"
#include "version.e"
#include "xbitmap.e"
#include "xpixmap.e"

#define CUR_VERSION 27
#define START_HAVING_ATTRS 8

extern char	* getenv ();

int	PRTGIF = FALSE;
char	curFileName[MAXPATHLENGTH+1];
int	curFileDefined = FALSE;
int	fileVersion = INVALID;
int	importingFile = FALSE;
int	psDotsPerInch = 72;
int	printMag = 100;
int	saveTmpOnReturn = TRUE;
int	warpToWinCenter = TRUE;

#ifndef A4PAPER
char	*psXOffStr[MAXPAGESTYLES] = {"0","0"};
float	psXOff[MAXPAGESTYLES] =     { 0,  0 };
char	*psYOffStr[MAXPAGESTYLES] = {"11","0"};
float	psYOff[MAXPAGESTYLES] =     { 11,  0 };
float	psPageWidthInInch[MAXPAGESTYLES]  = { 8.5, 11 };
float	psPageHeightInInch[MAXPAGESTYLES] = { 11, 8.5 };
#else /* A4PAPER */
char	*psXOffStr[MAXPAGESTYLES] = {"0","0"};
float	psXOff[MAXPAGESTYLES] =     { 0,  0 };
char	*psYOffStr[MAXPAGESTYLES] = {"11.7","0"};
float	psYOff[MAXPAGESTYLES] =     { 11.7,  0 };
float	psPageWidthInInch[MAXPAGESTYLES]  = { 8.25,11.7 };
float	psPageHeightInInch[MAXPAGESTYLES] = { 11.7,8.25 };
#endif /* A4PAPER */

char	printCommand[MAXSTRING+1];
char	outputDir[MAXPATHLENGTH+1];

char	* savedComments=NULL;
int	savedCommentsLen=0;
int	saveCommentsInSaveNew=TRUE;
int	usePsAdobeString=FALSE;

static
int CopyAFile (file1, file2)
   char	* file1, * file2;
{
   char	tmp_str[MAXSTRING+1], * rest;
   int	short_name;
   FILE	* fp1, * fp2;

   if ((fp1 = fopen (file1, "r")) == NULL)
   {
      sprintf (tmp_str, "Can not open '%s' for read.", file1);
      if (PRTGIF)
         fprintf (stderr, "%s\n", tmp_str);
      else
         Msg (tmp_str);
      return (FALSE);
   }
   if (short_name = IsPrefix (bootDir, file2, &rest)) ++rest;
   if ((fp2 = fopen (file2, "w")) == NULL)
   {
      if (PRTGIF)
         fprintf (stderr, "Can not open '%s' for write.\n", file2);
      else
      {
         if (short_name)
            sprintf (tmp_str, "Can not open '%s' for write.", rest);
         else
            sprintf (tmp_str, "Can not open '%s' for write.", file2);
         Msg (tmp_str);
      }
      fclose (fp1);
      return (FALSE);
   }
   while (fgets (tmp_str, MAXSTRING, fp1) != NULL) fputs (tmp_str, fp2);
   fclose (fp1);
   fclose (fp2);
   return (TRUE);
}

static
int ExecuteCmd (cmd)
   char	* cmd;
{
   char	tmp_str[MAXSTRING+1];
   FILE	* fp;

   return (FALSE);   /* disable this routine  tjb 11/10/92 */
   /***********
   if ((fp = popen (cmd, "r")) == NULL) return (FALSE); 

   while (fgets (tmp_str, MAXSTRING, fp) != NULL)
   {
      if (PRTGIF)
         fprintf (stderr, "%s", tmp_str);
      else
         Msg (tmp_str);
      sleep (5);
   }
    pclose (fp);
   return (TRUE);
   ************/
}

void ClearFileInfo ()
{
   curFileName[0] = '\0';
   curFileDefined = FALSE;
   *curSymDir = '\0';
}

void CleanUpComments ()
{
   if (savedComments != NULL)
   {
      free (savedComments);
      savedComments = NULL;
      savedCommentsLen = 0;
   }
}

int OkayToCreateFile (FileName)
   char	* FileName;
{
   FILE	* fp;

   if ((fp = fopen (FileName, "r")) == NULL) return (TRUE);
   fclose (fp);
   switch (YesNoCancel ("File exists, okay to overwrite? [ync](y)",CONFIRM_YES))
   {
      case CONFIRM_YES: break;
      case CONFIRM_NO: return (FALSE);
      case CONFIRM_CANCEL: return (FALSE);
   }
   unlink (FileName);
   return (TRUE);
}

void Save (FP, BotObjPtr, Level)
   FILE			* FP;
   struct ObjRec	* BotObjPtr;
   int			Level;
{
   struct ObjRec	* obj_ptr;

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   TieLooseEnds ();

   if (Level == 0)
   {
      if (TGIF_PATCHLEVEL == 0)
         fprintf (FP, "%%TGIF %s\n", version_string);
      else
         fprintf (FP, "%%TGIF %s-p%1d\n", version_string, TGIF_PATCHLEVEL);

      fprintf (FP, "state(%1d,%1d,%1d,", pageStyle, CUR_VERSION, printMag);
      fprintf (FP, "%1d,%1d,%1d,", drawOrigX, drawOrigY, zoomScale);
      fprintf (FP, "%1d,%1d,%1d,", xyEnglishGrid, gridOn, colorIndex);
      fprintf (FP, "%1d,%1d,%1d,", horiAlign, vertAlign, lineWidth);
      fprintf (FP, "%1d,%1d,%1d,%1d,", curSpline, lineStyle, objFill, penPat);
      fprintf (FP, "%1d,%1d,%1d,%1d,", textJust, curFont, curStyle, curSize);
      fprintf (FP, "%1d,%1d,%1d,", curFontDPI, curDash, gridSystem);
      fprintf (FP, "%1d,%1d,%1d,", xyMetricGrid, textVSpace, zoomedIn);
      fprintf (FP, "%1d,%1d,%1d,", gridShown, moveMode, curRotate);
      fprintf (FP, "%1d,%1d).\n", rcbRadius, useGray);

      fprintf (FP, "%%\n");
      fprintf (FP, "%% @%s%s\n", "(#)$H", "eader$");
      fprintf (FP, "%% %s\n", "%W%");
      fprintf (FP, "%%\n");

      if (savedComments != NULL) fputs (savedComments, FP);
   }

   for (obj_ptr = BotObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_POLY: SavePolyObj (FP, obj_ptr); break;
         case OBJ_BOX: SaveBoxObj (FP, obj_ptr); break;
         case OBJ_OVAL: SaveOvalObj (FP, obj_ptr); break;
         case OBJ_TEXT: SaveTextObj (FP, obj_ptr); break;
         case OBJ_POLYGON: SavePolygonObj (FP, obj_ptr); break;
         case OBJ_ARC: SaveArcObj (FP, obj_ptr); break;
         case OBJ_RCBOX: SaveRCBoxObj (FP, obj_ptr); break;
         case OBJ_XBM: SaveXBmObj (FP, obj_ptr); break;
         case OBJ_XPM: SaveXPmObj (FP, obj_ptr); break;
         case OBJ_GROUP: SaveGroupObj (FP, obj_ptr, Level); break;
         case OBJ_SYM: SaveCompObj (FP, obj_ptr, Level); break;
         case OBJ_ICON: SaveIconObj (FP, obj_ptr, Level); break;
      }
      if (obj_ptr->prev == NULL)
         if (Level == 0)
            fprintf (FP, ".\n");
         else
            fprintf (FP, "\n");
      else
         if (Level == 0)
            fprintf (FP, ".\n");
         else
            fprintf (FP, ",\n");
   }
   SetCurChoice (NOTHING);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);
}

int SaveTmpFile (NewFileName)
   char	* NewFileName;
   /* return TRUE if file successfully saved */
{
   char			new_file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1];
   char			* rest;
   FILE			* fp;
   int			count = 0, status, short_name;
   struct ObjRec	* obj_ptr;

   strcpy (new_file_name, NewFileName);

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;

   switch (count)
   {
      case 0:
         sprintf (new_file_name, "%s.%s", NewFileName, OBJ_FILE_EXT);
         status = OBJ_FILE_SAVED;
         break;
      case 1:
         sprintf (new_file_name, "%s.%s", NewFileName, SYM_FILE_EXT);
         status = SYM_FILE_SAVED;
         break;
      default:
         Msg ("TOO MANY SYMBOLS!  Symbol file not saved.");
         return (INVALID);
   }

   unlink (new_file_name);

   if (short_name = IsPrefix (bootDir, new_file_name, &rest)) ++rest;
   if ((fp = fopen (new_file_name, "w")) == NULL)
   {
      if (short_name)
         sprintf (s, "Can not open '%s', file not saved.", rest);
      else
         sprintf (s, "Can not open '%s', file not saved.", new_file_name);
      Msg (s);
      return (INVALID);
   }

   if (short_name)
      sprintf (s, "Saving temporary file '%s' ...", rest);
   else
      sprintf (s, "Saving temporary file '%s' ...", new_file_name);
   Msg (s);
   Save (fp, botObj, 0);
   fclose (fp);
   if (short_name)
      sprintf (s, "Temporary file '%s' saved.", rest);
   else
      sprintf (s, "Temporary file '%s' saved.", new_file_name);
   Msg (s);

   if (PSFILE_MOD != 0 && chmod (new_file_name, 0777))
   {
      if (short_name)
         sprintf (s, "Can not chmod '%s' to 0777.", rest);
      else
         sprintf (s, "Can not chmod '%s' to 0777.", new_file_name);
      Msg (s);
   }
   return (status);
}

void SaveNewFile (SaveSelectedOnly)
   int	SaveSelectedOnly;
{
   char			new_file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1];
   char			new_full_name[MAXPATHLENGTH+1];
   char			tmp_str[MAXPATHLENGTH+1];
   char			name_without_ext[MAXPATHLENGTH+1], * rest;
   char			saved_cur_dir[MAXPATHLENGTH+1];
   char			saved_cur_file_name[MAXPATHLENGTH+1];
   char			obj_ext_str[MAXSTRING+1], sym_ext_str[MAXSTRING+1];
   int			saved_cur_file_defined=FALSE, obj_ext_len, sym_ext_len;
   FILE			* fp;
   int			count = 0, len, short_name;
   struct ObjRec	* obj_ptr, * saved_top_obj=NULL, * saved_bot_obj=NULL;
   struct SelRec	* top_sel_ptr, * bot_sel_ptr;
   struct SelRec	* sel_ptr, * next_sel;

   if (SaveSelectedOnly && topSel==NULL)
   {
      Msg ("No object selected!  Nothing saved!");
      return;
   }

   Dialog ("Please enter new file name:", "( <CR>: accept, <ESC>: cancel )",
         new_file_name);
   if (*new_file_name == '\0') return;
   len = strlen (new_file_name);

   if (SaveSelectedOnly)
   {
      for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
         if (sel_ptr->obj->type == OBJ_SYM) count++;
   }
   else
   {
      for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
         if (obj_ptr->type == OBJ_SYM) count++;
   }

   sprintf (obj_ext_str, ".%s", OBJ_FILE_EXT);
   sprintf (sym_ext_str, ".%s", SYM_FILE_EXT);
   obj_ext_len = strlen (obj_ext_str);
   sym_ext_len = strlen (sym_ext_str);

   switch (count)
   {
      case 0:
         if (len >= obj_ext_len || len >= sym_ext_len)
         {
            if (strcmp (&new_file_name[len-sym_ext_len], sym_ext_str) == 0)
            {
               Msg ("Can not save as a symbol file, no symbol defined.");
               return;
            }
            else if (strcmp (&new_file_name[len-obj_ext_len], obj_ext_str) != 0)
            {
               strcpy (name_without_ext, new_file_name);
               strcat (new_file_name, obj_ext_str);
            }
            else
            {
               strcpy (name_without_ext, new_file_name);
               name_without_ext[len-obj_ext_len] = '\0';
            }

            if (strlen (new_file_name) == obj_ext_len)
            {
               Msg ("No file name specified.  File not saved.");
               return;
            }
         }
         else
         {
            strcpy (name_without_ext, new_file_name);
            strcat (new_file_name, obj_ext_str);
         }
         break;
      case 1:
         if (len >= obj_ext_len || len >= sym_ext_len)
         {
            if (strcmp (&new_file_name[len-obj_ext_len], obj_ext_str) == 0)
            {
               Msg ("Can not save as an object file; ther's a symbol defined.");
               return;
            }
            else if (strcmp (&new_file_name[len-sym_ext_len], sym_ext_str) != 0)
            {
               strcpy (name_without_ext, new_file_name);
               strcat (new_file_name, sym_ext_str);
            }
            else
            {
               strcpy (name_without_ext, new_file_name);
               name_without_ext[len-sym_ext_len] = '\0';
            }

            if (strlen (new_file_name) == sym_ext_len)
            {
               Msg ("No file name specified.  File not saved.");
               return;
            }
         }
         else
         {
            strcpy (name_without_ext, new_file_name);
            strcat (new_file_name, sym_ext_str);
         }
         break;
      default:
         Msg ("TOO MANY SYMBOLS!  Symbol file not saved.");
         return;
   }

   if (*new_file_name == '/')
      strcpy (new_full_name, new_file_name);
   else
      sprintf (new_full_name, "%s/%s", curDir, new_file_name);

   if (!OkayToCreateFile (new_full_name)) return;

   if (short_name = IsPrefix (bootDir, new_full_name, &rest)) ++rest;
   if ((fp = fopen (new_full_name, "w")) == NULL)
   {
      if (short_name)
         sprintf (s, "Can not open '%s', file not saved.", rest);
      else
         sprintf (s, "Can not open '%s', file not saved.", new_full_name);
      Msg (s);
      return;
   }

   if (SaveSelectedOnly)
   {
      strcpy (saved_cur_dir, curDir);
      strcpy (saved_cur_file_name, curFileName);
      saved_cur_file_defined = curFileDefined;
      saved_top_obj = topObj;
      saved_bot_obj = botObj;

      JustDupSelObj (&top_sel_ptr, &bot_sel_ptr);
      topObj = top_sel_ptr->obj;
      botObj = bot_sel_ptr->obj;

      for (sel_ptr=topSel, obj_ptr=topObj; obj_ptr!=NULL;
            sel_ptr=sel_ptr->next, obj_ptr=obj_ptr->next)
      {
         CopyObjId (sel_ptr->obj, obj_ptr);
         CopyObjLocks (sel_ptr->obj, obj_ptr);
      }
   }

   strcpy (tmp_str, curDir);
   SetCurDir (new_full_name);
   curFileDefined = TRUE;

   switch (count)
   {
      case 0:
         *curSymDir = '\0';
         if ((strcmp (tmp_str, curDir) != 0) || (!NameInCurDir (curFileName)))
            UpdateDirInfo ();
         break;
      case 1:
         strcpy (curSymDir, curDir);
         if (!DirInSymPath (curDir)) UpdateSymInfo ();
         break;
   }

   if (short_name)
      sprintf (s, "Saving '%s' ...", rest);
   else
      sprintf (s, "Saving '%s' ...", new_full_name);
   Msg (s);

   if (!saveCommentsInSaveNew) CleanUpComments ();
   Save (fp, botObj, 0);
   fclose (fp);

   if (short_name)
      sprintf (s, "File '%s' saved.", rest);
   else
      sprintf (s, "File '%s' saved.", new_full_name);
   Msg (s);

   if (SaveSelectedOnly)
   {
      DelAllObj ();
      for (sel_ptr = top_sel_ptr; sel_ptr != NULL; sel_ptr = next_sel)
      {
         next_sel = sel_ptr->next;
         free (sel_ptr);
      }

      topObj = saved_top_obj;
      botObj = saved_bot_obj;
      strcpy (curDir, saved_cur_dir);
      strcpy (curFileName, saved_cur_file_name);
      curFileDefined = saved_cur_file_defined;
   }
   else
   {
      SetFileModified (FALSE);
      RedrawTitleWindow ();
   }
}

void SaveSymInLibrary ()
{
   char			new_file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1];
   char			new_full_name[MAXPATHLENGTH+1];
   char			dir_name[MAXPATHLENGTH+1];
   char			saved_dir[MAXPATHLENGTH+1], saved_file[MAXPATHLENGTH+1];
   char			saved_sym_dir[MAXPATHLENGTH+1];
   char			name_without_ext[MAXPATHLENGTH+1], * rest;
   char			sym_ext_str[MAXSTRING+1], * c_ptr;
   FILE			* fp;
   int			count=0, len, short_name, sym_ext_len;
   int			saved_cur_file_defined;
   struct ObjRec	* obj_ptr;

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;

   switch (count)
   {
      case 0: Msg ("No Symbol found!  Symbol file not saved."); return;
      case 1: break;
      default: Msg ("TOO MANY SYMBOLS!  Symbol file not saved."); return;
   }

   sprintf (sym_ext_str, ".%s", SYM_FILE_EXT);
   sym_ext_len = strlen (sym_ext_str);

   if (*curFileName == '\0')
   {
      Dialog ("Please enter new file name:", "( <CR>: accept, <ESC>: cancel )",
            new_file_name);
      if (*new_file_name == '\0') return;

      len = strlen (new_file_name);
      if (len >= sym_ext_len)
      {
         if (strcmp (&new_file_name[len-sym_ext_len], sym_ext_str) != 0)
         {
            strcpy (name_without_ext, new_file_name);
            strcat (new_file_name, sym_ext_str);
         }
         else
         {
            strcpy (name_without_ext, new_file_name);
            name_without_ext[len-sym_ext_len] = '\0';
         }
   
         if (strlen (new_file_name) == sym_ext_len)
         {
            Msg ("No file name specified.  File not saved.");
            return;
         }
      }
      else
      {
         strcpy (name_without_ext, new_file_name);
         strcat (new_file_name, sym_ext_str);
      }
   }
   else
   {
      len = strlen (curFileName);
      for (c_ptr = &curFileName[len-1]; c_ptr != curFileName; c_ptr--)
         if (*c_ptr == '/')
            break;
      if (*c_ptr == '/') c_ptr++;
      strcpy (new_file_name, c_ptr);
   }

   if (SelectSymDir (dir_name) == INVALID) { Msg (""); return; }

   if (strcmp (dir_name, ".") == 0)
      sprintf (new_full_name, "%s/%s", curDir, new_file_name);
   else
      sprintf (new_full_name, "%s/%s", dir_name, new_file_name);

   if (!OkayToCreateFile (new_full_name)) return;

   if (short_name = IsPrefix (bootDir, new_full_name, &rest)) ++rest;
   if ((fp = fopen (new_full_name, "w")) == NULL)
   {
      if (short_name)
         sprintf (s, "Can not open '%s', file not saved.", rest);
      else
         sprintf (s, "Can not open '%s', file not saved.", new_full_name);
      Msg (s);
      return;
   }

   strcpy (saved_dir, curDir);
   strcpy (saved_file, curFileName);
   strcpy (saved_sym_dir, curSymDir);
   saved_cur_file_defined = curFileDefined;
   SetCurDir (new_full_name);
   curFileDefined = TRUE;

   strcpy (curSymDir, curDir);
   if (!DirInSymPath (curDir)) UpdateSymInfo ();

   if (short_name)
      sprintf (s, "Saving '%s' ...", rest);
   else
      sprintf (s, "Saving '%s' ...", new_full_name);
   Msg (s);

   if (!saveCommentsInSaveNew) CleanUpComments ();
   Save (fp, botObj, 0);
   fclose (fp);

   if (short_name)
      sprintf (s, "File '%s' saved.", rest);
   else
      sprintf (s, "File '%s' saved.", new_full_name);
   Msg (s);

   strcpy (curDir, saved_dir);
   strcpy (curFileName, saved_file);
   strcpy (curSymDir, saved_sym_dir);
   curFileDefined = saved_cur_file_defined;

   RedrawTitleWindow ();
}

void SaveFile ()
{
   int			i, len, count, short_name;
   struct ObjRec	* obj_ptr;
   FILE			* fp;
   char			ext[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1];
   char			full_name[MAXPATHLENGTH+1], * rest;

   if (!curFileDefined)
   {
      SaveNewFile (FALSE);
      return;
   }

   len = strlen (curFileName);
   for (i = len-1; curFileName[i] != '.'; i--) ;
   strcpy (ext, &curFileName[i+1]);

   for (obj_ptr = topObj, count = 0; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;
   switch (count)
   {
      case 0:
         if (strcmp (ext, SYM_FILE_EXT) == 0)
         {
            Msg ("No SYMBOL defined in symbol file.  Symbol file not saved.");
            return;
         }
         break;
      case 1:
         if (strcmp (ext, OBJ_FILE_EXT) == 0)
         {
            Msg ("One SYMBOL defined in OBJECT file.  Object file not saved.");
            return;
         }
         break;
      default:
         if (strcmp (ext, SYM_FILE_EXT) == 0)
         {
            Msg ("Too many SYMBOLS in symbol file!  Symbol file not saved.");
            return;
         }
         break;
   }

   if (strcmp (ext, SYM_FILE_EXT) == 0)
      sprintf (full_name, "%s/%s", curSymDir, curFileName);
   else if (strcmp (ext, OBJ_FILE_EXT) == 0)
      sprintf (full_name, "%s/%s", curDir, curFileName);

   if (short_name = IsPrefix (bootDir, full_name, &rest)) ++rest;
   if ((fp = fopen (full_name, "w")) == NULL)
   {
      if (short_name)
         sprintf (s, "Can not open '%s', file not saved.", rest);
      else
         sprintf (s, "Can not open '%s', file not saved.", full_name);
      Msg (s);
      return;
   }

   if (short_name)
      sprintf (s, "Saving '%s' ...", rest);
   else
      sprintf (s, "Saving '%s' ...", full_name);
   Msg (s);

   Save (fp, botObj, 0);

   fclose (fp);
   if (short_name)
      sprintf (s, "File '%s' saved.", rest);
   else
      sprintf (s, "File '%s' saved.", full_name);
   Msg (s);
   SetFileModified (FALSE);
   if (!NameInCurDir (curFileName)) UpdateDirInfo ();
}

char * ParseStr (Str, C, Left)
   char	* Str, C, * Left;
{
   register char	* s = Str, * l = Left;
   register int		len = 0;

   while (*s != '\0' && *s != C)
   {
      *l++ = *s++;
      len++;
   }

   if (*s == C) s++;
   *l = '\0';

   while (len >= 2 && *Left == '\'' && *(l-1) == '\'')
   {
      *(--l) = '\0';
      strcpy (Left, &Left[1]);
      len -= 2;
   }
   return (s);
}

char * FindChar (C, Str)
   char	C;
   char	* Str;
   /* returns the address of the character right after C of the string Str */
{
   register char	* s = Str;

   while (*s != '\0' && *s != C) s++;

   if (*s == C) s++;
   return (s);
}

#define GETVALUE(val,name) \
      if (ScanValue("%d", (char *) &(val), name, "state") == INVALID) \
         return (FALSE)

static
int ReadState (Inbuf)
   char	* Inbuf;
{
   char	* s;
   int	page_style, forced_use_gray=FALSE;

   s = FindChar ('(', Inbuf);
   if (sscanf (s, "%d", &page_style) != 1) return (FALSE);
   s = FindChar (',', s);
   if (*s == '\0')
      fileVersion = INVALID;
   else if (sscanf (s, "%d", &fileVersion) != 1)
      return (FALSE);

   if (fileVersion > CUR_VERSION) return (FALSE);

   if (!importingFile)
   {
      if (fileVersion <= 13)
      {
         switch (page_style)
         {
            case PORTRAIT: printMag = 100; break;
            case LANDSCAPE: printMag = 100; break;
            case HIGHPORT: printMag = 50; page_style = PORTRAIT; break;
            case HIGHLAND: printMag = 50; page_style = LANDSCAPE; break;
            case SLIDEPORT: printMag = 200; page_style = PORTRAIT; break;
            case SLIDELAND: printMag = 200; page_style = LANDSCAPE; break;
         }
      }
      else
      {
         s = FindChar(',', s);
         sscanf (s, "%d", &printMag);
      }
      pageStyle = page_style;
   }

   if (PRTGIF && useGray) forced_use_gray = TRUE;
   if (!importingFile)
   {
      if (fileVersion >= 2)
      {
         curFontDPI = FONT_DPI_75;
         curDash = 0;
         gridSystem = ENGLISH_GRID;
         xyMetricGrid = DEFAULT_METRIC_GRID;
         textVSpace = 0;
         zoomedIn = FALSE;
         curRotate = ROTATE0;
         rcbRadius = DEF_RCB_RADIUS;

         s = FindChar (',', s);

         InitScan (s, "\t\n, ");

         if (fileVersion <= 3)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");

            if (lineWidth == LINE_CURVED)
            {
               lineWidth = 0;
               curSpline = LT_SPLINE;
            }
            else
               curSpline = LT_STRAIGHT;
         }
         else if (fileVersion <= 7)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
         }
         else if (fileVersion <= 8)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
         }
         else if (fileVersion <= 11)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
         }
         else if (fileVersion <= 12)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
         }
         else if (fileVersion <= 18)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
            GETVALUE(textVSpace,    "Text Vertical Spacing");
         }
         else if (fileVersion <= 19)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
            GETVALUE(textVSpace,    "Text Vertical Spacing");
            GETVALUE(zoomedIn,      "Zoomed In");
         }
         else if (fileVersion <= 21)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
            GETVALUE(textVSpace,    "Text Vertical Spacing");
            GETVALUE(zoomedIn,      "Zoomed In");
            GETVALUE(gridShown,     "Grid Shown");
            GETVALUE(moveMode,      "Move Mode");
         }
         else if (fileVersion <= 26)
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
            GETVALUE(textVSpace,    "Text Vertical Spacing");
            GETVALUE(zoomedIn,      "Zoomed In");
            GETVALUE(gridShown,     "Grid Shown");
            GETVALUE(moveMode,      "Move Mode");
            GETVALUE(curRotate,     "Text Rotation");
            GETVALUE(rcbRadius,     "RCBox Radius");
         }
         else
         {
            GETVALUE(drawOrigX,     "X Draw Origin");
            GETVALUE(drawOrigY,     "Y Draw Origin");
            GETVALUE(zoomScale,     "Zoom scale");
            GETVALUE(xyEnglishGrid, "English Grid");
            GETVALUE(gridOn,        "Grid");
            GETVALUE(colorIndex,    "Color");
            GETVALUE(horiAlign,     "Horizontal Align");
            GETVALUE(vertAlign,     "Vertical Align");
            GETVALUE(lineWidth,     "Line Width");
            GETVALUE(curSpline,     "Spline");
            GETVALUE(lineStyle,     "Line Style");
            GETVALUE(objFill,       "Fill Pattern");
            GETVALUE(penPat,        "Pen Pattern");
            GETVALUE(textJust,      "Text Justify");
            GETVALUE(curFont,       "Font Name");
            GETVALUE(curStyle,      "Font Style");
            GETVALUE(curSize,       "Font Size");
            GETVALUE(curFontDPI,    "Font DPI");
            GETVALUE(curDash,       "Dash Style");
            GETVALUE(gridSystem,    "Grid System");
            GETVALUE(xyMetricGrid,  "Metric Grid");
            GETVALUE(textVSpace,    "Text Vertical Spacing");
            GETVALUE(zoomedIn,      "Zoomed In");
            GETVALUE(gridShown,     "Grid Shown");
            GETVALUE(moveMode,      "Move Mode");
            GETVALUE(curRotate,     "Text Rotation");
            GETVALUE(rcbRadius,     "RCBox Radius");
            GETVALUE(useGray,       "Use Gray Scale");
         }
         if (PRTGIF)
         {
            if (forced_use_gray) useGray = TRUE;
            return (TRUE);
         }

         if (colorIndex >= maxColors)
         {
            fprintf (stderr, "%s #%1d, use '%s' %s.\n",
                  "In reading state, can not find color", colorIndex,
                  colorMenuItems[defaultColorIndex], "as the current color");
            colorIndex = defaultColorIndex;
         }
         SetCanvasFont ();
         if (fileVersion <= 13)
         {
            switch (gridSystem)
            {
               case ENGLISH_GRID:
                  drawOrigX += HALF_INCH;
                  drawOrigY += HALF_INCH;
                  break;
               case METRIC_GRID:
                  drawOrigX += 2.5*ONE_CM;
                  drawOrigY += 2.5*ONE_CM;
                  break;
            }
         }
      }
      if (PRTGIF) return (TRUE);

      if (lineWidth >= maxLineWidths)
      {
         fprintf (stderr, "%s '%1d' is out of range!  Set to 0.\n",
               "File's linewidth index", lineWidth);
         lineWidth = 0;
      }

      UpdDrawWinWH ();
      UpdPageStyle (pageStyle);
      UpdDrawWinBBox ();

      DrawPaperBoundary ();
      RedrawGridLines ();
      RedrawRulers ();
      RedrawChoiceWindow ();
   }
   return (TRUE);
}

static
void ReadObjAttrs (MinFileVersion, FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* * ObjPtr;
{
   struct AttrRec	* top_attr = NULL, * bot_attr = NULL, * attr_ptr;

   if (fileVersion <= MinFileVersion) return;

   while (ReadAttr (FP, &attr_ptr))
   {
      attr_ptr->owner = *ObjPtr;
      attr_ptr->prev = NULL;
      attr_ptr->next = top_attr;
      if (top_attr == NULL)
         bot_attr = attr_ptr;
      else
         top_attr->prev = attr_ptr;
      top_attr = attr_ptr;
   }
   if (bot_attr != NULL) bot_attr->next = NULL;
   (*ObjPtr)->fattr = top_attr;
   (*ObjPtr)->lattr = bot_attr;
}

int ReadObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* * ObjPtr;
{
   char			inbuf[MAXSTRING+1], obj_name[10];
   int			len, cur_size, done, allocated, read_state_ok;

   *ObjPtr = NULL;
   while (fgets (inbuf, MAXSTRING, FP) != NULL)
   {
      char	* line, * c_ptr;

      scanLineNum++;
      if (inbuf[0] == ']') return (FALSE);
      allocated = FALSE;

      len = strlen(inbuf);
      if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
      {  /* line longer than MAXSTRING characters */
         /* inbuf[MAXSTRING-1] == '\0' and len == MAXSTRING-1 now */
         cur_size = 2*MAXSTRING-1;
         line = (char *) calloc (cur_size+1, sizeof(char));
         strcpy (line, inbuf);
         c_ptr = &(line[MAXSTRING-1]);
         allocated = TRUE;

         done = FALSE;
         while (!done && fgets (inbuf, MAXSTRING, FP) != NULL)
         {
            scanLineNum++;
            len = strlen(inbuf);
            if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
            {
               done = TRUE;
               inbuf[len-1] = '\0';
               strcpy (c_ptr, inbuf);
            }
            else
            {
               int	n = c_ptr - line;	/* NMH */

               cur_size += MAXSTRING-1;
               line = (char *) realloc (line, cur_size+1);
               c_ptr = line + n;		/* NMH */
               strcpy (c_ptr, inbuf);
               c_ptr += MAXSTRING-1;
            }
         }
         if (!done)
         {
            sprintf (inbuf, "%s, %d:  EOF in ReadObj ().  Read aborted!",
                  scanFileName, scanLineNum);
            if (PRTGIF)
               fprintf (stderr, "%s\n", inbuf);
            else
               Msg (inbuf);
            if (allocated) free (line);
            return (FALSE);
         }
      }
      else
      {
         line = inbuf;
         line[len-1] = '\0';
      }

      if (line[0] == '%')
      {
         if (!importingFile && line[1]=='%')
         {
            int	line_len = strlen (line);

            if (savedComments == NULL)
            {
               if ((savedComments = (char *) calloc (line_len+2, sizeof(char)))
                     == NULL)
                  Error ("ReadObj()", "Can not calloc()");
               *savedComments = '\0';
            }
            else
            {
               if ((savedComments = (char *) realloc (savedComments,
                     savedCommentsLen+line_len+2)) == NULL)
                  Error ("ReadObj()", "Can not realloc()");
               savedComments[savedCommentsLen] = '\0';
            }
            strcat (savedComments, line);
            savedCommentsLen += line_len;
            savedComments[savedCommentsLen++] = '\n';
            savedComments[savedCommentsLen] = '\0';
         }
         continue;
      }

      ParseStr (line, '(', obj_name);
      if (strcmp (obj_name, "poly") == 0)
      {
         ReadPolyObj (FP, line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (INVALID, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "box") == 0)
      {
         ReadBoxObj (line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "oval") == 0)
      {
         ReadOvalObj (line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "text") == 0)
      {
         ReadTextObj (FP, line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "polygon") == 0)
      {
         ReadPolygonObj (FP, line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "arc") == 0)
      {
         ReadArcObj (line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "rcbox") == 0)
      {
         ReadRCBoxObj (line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "xbm") == 0)
      {
         ReadXBmObj (FP, line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "xpm") == 0)
      {
         ReadXPmObj (FP, line, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "group") == 0)
      {
         ReadGroupObj (FP, OBJ_GROUP, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (INVALID, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "sym") == 0)
      {
         ReadGroupObj (FP, OBJ_SYM, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (INVALID, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "icon") == 0)
      {
         ReadGroupObj (FP, OBJ_ICON, ObjPtr);
         if (*ObjPtr == NULL) return (FALSE);
         ReadObjAttrs (INVALID, FP, ObjPtr);
         AdjObjBBox (*ObjPtr);
         if (allocated) free (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "state") == 0)
      {
         read_state_ok = ReadState (line);
         *ObjPtr = NULL;
         if (allocated) free (line);
         return ((read_state_ok) ? TRUE : INVALID);
      }
      if (allocated) free (line);
   }
   return (FALSE);
}

static
void ConvertToUpperCase (InStr, OutStr)
   register char	* InStr, * OutStr;
{
   for ( ; *InStr != '\0'; InStr++, OutStr++)
      *OutStr = (*InStr>='a' && *InStr<='z') ? *InStr-'a'+'A' : *InStr;
   *OutStr = '\0';
}

void ChangeDomain ()
{
   char 	domain_name[MAXPATHLENGTH+1], env_str[MAXPATHLENGTH+1];
   char 	s[MAXSTRING+1], s1[MAXSTRING+1], * c_ptr;
   char 	cap_tool_name[MAXSTRING+1];
   XEvent	ev;

   if (SelectDomain (domain_name) == INVALID) return;

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   ConvertToUpperCase (TOOL_NAME, cap_tool_name);
   sprintf (env_str, "%s_%s", cap_tool_name, domain_name);
   if ((c_ptr = getenv (env_str)) == NULL)
   {
      if (strcmp (domain_name, "Examples") == 0)
         ParseSymPath (TGIF_PATH);
      else
         ParseSymPath (".");
   }
   else
      ParseSymPath (c_ptr);

   UpdateSymInfo ();

   strcpy (curDomainName, domain_name);
   sprintf (s, "Current domain is '%s'.", curDomainName);
   sprintf (s1, "Symbol path set to '%s'.", curDomainPath);
   TwoLineMsg (s, s1);
   RedrawTitleWindow ();
}

void AdjForOldVersion (obj_ptr)
   struct ObjRec	* obj_ptr;
{
   if (fileVersion <= 13)
   {
      switch (gridSystem)
      {
         case ENGLISH_GRID:
            MoveObj (obj_ptr, (int)(HALF_INCH), (int)(HALF_INCH));
            break;
         case METRIC_GRID:
            MoveObj (obj_ptr, (int)(2.5*ONE_CM), (int)(2.5*ONE_CM));
            break;
      }
   }
}

void ImportFile ()
{
   struct ObjRec	* obj_ptr, * saved_top_obj, * saved_bot_obj;
   char 		file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1], * rest;
   FILE			* fp;
   int			short_name, read_status;
   XEvent		ev;
   char 		tmp_filename[MAXPATHLENGTH+1];
   int 			tmp_linenum;

   MakeQuiescent ();

   importingFile = TRUE;
   if (importFromLibrary)
   {
      char	name[MAXSTRING+1], path[MAXSTRING+1];

      if (SelectFromLibrary("Please select an object file to IMPORT...",
            OBJ_FILE_EXT, name, path) == INVALID)
      {
         importingFile = FALSE;
         return;
      }
      sprintf (file_name, "%s/%s", path, name);
   }
   else if (SelectFileNameToImport("Please select an object file to IMPORT...",
         OBJ_FILE_EXT, file_name) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
   if ((fp = fopen (file_name, "r")) == NULL)
   {
      if (short_name)
         sprintf (s, "Can not import '%s'.", rest);
      else
         sprintf (s, "Can not import '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   strcpy (tmp_filename, scanFileName);
   tmp_linenum = scanLineNum;
   strcpy (scanFileName, (short_name ? rest : file_name));
   scanLineNum = 0;

   saved_top_obj = topObj;
   saved_bot_obj = botObj;
   topObj = botObj = NULL;

   if (short_name)
      sprintf (s, "Importing '%s' ...", rest);
   else
      sprintf (s, "Importing '%s' ...", file_name);
   Msg (s);

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   while ((read_status = ReadObj (fp, &obj_ptr)) == TRUE)
      if (obj_ptr != NULL)
      {
         AdjForOldVersion (obj_ptr);
         UnlockAnObj (obj_ptr);
         AddObj (NULL, topObj, obj_ptr);
      }

   fclose (fp);
   importingFile = FALSE;

   strcpy (scanFileName, tmp_filename);
   scanLineNum = tmp_linenum;

   if (read_status == INVALID)
   {
      sprintf (s, "File version too large (=%1d).  Import aborted!",
            fileVersion);
      Msg (s);
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);
      return;
   }

   if (topObj != NULL) SetFileModified (TRUE);
   justDupped = FALSE;

   SelAllObj (FALSE);

   if (botObj != NULL)
      botObj->next = saved_top_obj;
   else
      topObj = saved_top_obj;

   if (saved_top_obj != NULL)
   {
      saved_top_obj->prev = botObj;
      botObj = saved_bot_obj;
   }

   PrepareToRecord (CMD_NEW, NULL, NULL, 0);
   RecordCmd (CMD_NEW, NULL, topSel, botSel, numObjSelected);
   RedrawDrawWindow (botObj);
   HighLightForward ();

   if (!importFromLibrary) SetCurImportDir (file_name);

   if (short_name)
      sprintf (s, "'%s' imported.", rest);
   else
      sprintf (s, "'%s' imported.", file_name);
   Msg (s);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);
}

void ImportXBitmapFile ()
{
   char 		file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1], * rest;
   char 		mag_spec[MAXSTRING+1], msg[MAXSTRING+1];
   unsigned int		tmp_w, tmp_h;
   int			rc, x_hot, y_hot, x, y, w, h, short_name;
   int			orig_w, orig_h;
   float		mag;
   Pixmap		orig_bitmap, bitmap;
   XImage		* image=NULL;
   XEvent		ev;
   struct ObjRec	* obj_ptr;

   MakeQuiescent ();

   importingFile = TRUE;
   if (importFromLibrary)
   {
      char	name[MAXSTRING+1], path[MAXSTRING+1];

      if (SelectFromLibrary("Please select an XBitmap file to IMPORT...",
            XBM_FILE_EXT, name, path) == INVALID)
      {
         importingFile = FALSE;
         return;
      }
      sprintf (s, "%s/%s", path, name);
   }
   else if (SelectFileNameToImport("Please select an XBitmap file to IMPORT...",
         XBM_FILE_EXT, s) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   strcpy (file_name, s);

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   rc = XReadBitmapFile (mainDisplay, mainWindow, file_name, &tmp_w, &tmp_h,
         &orig_bitmap, &x_hot, &y_hot);
   orig_w = tmp_w; orig_h = tmp_h;
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
   if (rc != BitmapSuccess)
   {
      if (short_name)
         sprintf (s, "Can not import XBitmap file '%s'.", rest);
      else
         sprintf (s, "Can not import XBitmap file '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   x = 0;
   y = 0;
   w = orig_w;
   h = orig_h;
   mag = 1.0;
   if (askForXBmSpec)
   {
      sprintf (msg, "%s: [[MAG=]WxH+X+Y] (original size is %1dx%1d)",
         "Please enter geometry spec", orig_w, orig_h);
      Dialog (msg, "( <CR>: accept, <ESC>: continue )", mag_spec);
      if (*mag_spec != '\0')
         ParseCutSpec (mag_spec, orig_w, orig_h, &mag, &x, &y, &w, &h);

      if (x==0 && y==0 && w==orig_w && h==orig_h && mag==1.0)
         bitmap = orig_bitmap;
      else
      {
         orig_w = w;
         orig_h = h;
         if (!ExtractBitmap (orig_bitmap, NULL, x, y, w, h, &bitmap, &image))
         {
            Msg ("Can not allocate extracted bitmap.");
            importingFile = FALSE;
            XFreePixmap (mainDisplay, orig_bitmap);
            return;
         }
         XFreePixmap (mainDisplay, orig_bitmap);
         w = (int)(((float)w) * mag);
         h = (int)(((float)h) * mag);
      }
   }
   else
      bitmap = orig_bitmap;

   obj_ptr = CreateXBmObj (orig_w, orig_h, w, h, bitmap, image);
   PlaceTopObj (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);

   SelectTopObj ();
   RecordNewObjCmd ();
   SetFileModified (TRUE);
   justDupped = FALSE;

   if (!importFromLibrary) SetCurImportDir (file_name);

   if (short_name)
      sprintf (s, "XBitmap file (%1dx%1d) '%s' imported.", orig_w, orig_h,
            rest);
   else
      sprintf (s, "XBitmap file (%1dx%1d) '%s' imported.", orig_w, orig_h,
            file_name);
   Msg (s);
   importingFile = FALSE;
}

void ImportXPixmapFile ()
{
   char 		file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1], * rest;
   int			rc, ncolors, chars_per_pixel, * pixels, short_name;
   int			first_pixel_is_bg, image_w, image_h, w, h;
   Pixmap		pixmap;
   XImage		* image;
   char			* color_char, * * color_str;
   XEvent		ev;
   struct ObjRec	* obj_ptr;

   MakeQuiescent ();

   importingFile = TRUE;
   if (importFromLibrary)
   {
      char	name[MAXSTRING+1], path[MAXSTRING+1];

      if (SelectFromLibrary("Please select an XPixmap file to IMPORT...",
            XPM_FILE_EXT, name, path) == INVALID)
      {
         importingFile = FALSE;
         return;
      }
      sprintf (s, "%s/%s", path, name);
   }
   else if (SelectFileNameToImport("Please select an XPixmap file to IMPORT...",
         XPM_FILE_EXT, s) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   strcpy (file_name, s);

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   rc = MyReadPixmapFile (file_name, &image_w, &image_h, &w, &h, &pixmap,
         &image, &ncolors, &chars_per_pixel, &first_pixel_is_bg, &color_char,
         &color_str, &pixels);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
   if (rc != BitmapSuccess)
   {
      if (short_name)
         sprintf (s, "Can not import XPixmap file '%s'.", rest);
      else
         sprintf (s, "Can not import XPixmap file '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   obj_ptr = CreateXPmObj (image_w, image_h, w, h, pixmap, image, ncolors,
         chars_per_pixel, first_pixel_is_bg, color_char, color_str, pixels);
   PlaceTopObj (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);

   SelectTopObj ();
   RecordNewObjCmd ();
   SetFileModified (TRUE);
   justDupped = FALSE;

   if (!importFromLibrary) SetCurImportDir (file_name);

   if (short_name)
      sprintf (s, "XPixmap file (%1dx%1d) '%s' imported.", image_w, image_h,
            rest);
   else
      sprintf (s, "XPixmap file (%1dx%1d) '%s' imported.", image_w, image_h,
            file_name);
   Msg (s);
   importingFile = FALSE;
}

void ImportEPSFile ()
{
   char 		file_name[MAXPATHLENGTH+1], s[MAXPATHLENGTH+1], * rest;
   char 		* * lines=NULL, write_date[32];
   int			rc, short_name, num_lines, epsf_level, image_w, image_h;
   float		llx, lly, urx, ury;
   Pixmap		bitmap;
   XImage		* image=NULL;
   XEvent		ev;
   struct ObjRec	* obj_ptr;

   MakeQuiescent ();

   importingFile = TRUE;
   if (importFromLibrary)
   {
      char	name[MAXSTRING+1], path[MAXSTRING+1];

      if (SelectFromLibrary("Please select an EPS file to IMPORT...",
            EPSF_FILE_EXT, name, path) == INVALID)
      {
         importingFile = FALSE;
         return;
      }
      sprintf (s, "%s/%s", path, name);
   }
   else if (SelectFileNameToImport("Please select an EPS file to IMPORT...",
         EPSF_FILE_EXT, s) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   strcpy (file_name, s);

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   rc = MyReadEPSFile (file_name, &image_w, &image_h, &bitmap, &image,
         &num_lines, &lines, &epsf_level, &llx, &lly, &urx, &ury, write_date);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
   if (rc != BitmapSuccess)
   {
      if (short_name)
         sprintf (s, "Can not import EPS file '%s'.", rest);
      else
         sprintf (s, "Can not import EPS file '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   if (short_name)
      obj_ptr = CreateEPSObj (rest, image_w, image_h, bitmap, image,
            num_lines, lines, epsf_level, llx, lly, urx, ury, write_date);
   else
      obj_ptr = CreateEPSObj (file_name, image_w, image_h, bitmap, image,
            num_lines, lines, epsf_level, llx, lly, urx, ury, write_date);

   PlaceTopObj (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);

   SelectTopObj ();
   RecordNewObjCmd ();
   SetFileModified (TRUE);
   justDupped = FALSE;

   if (!importFromLibrary) SetCurImportDir (file_name);

   if (short_name)
      sprintf (s, "EPS file '%s' imported.", rest);
   else
      sprintf (s, "EPS file '%s' imported.", file_name);
   Msg (s);
   importingFile = FALSE;
}

int LoadFile (FullName)
   char	* FullName;
{
   struct ObjRec	* obj_ptr;
   char 		file_name[MAXPATHLENGTH+1], msg[MAXPATHLENGTH+1];
   char 		saved_cur_dir[MAXPATHLENGTH+1], * rest;
   int			read_status, short_name;
   FILE			* fp;
   char 		tmp_filename[MAXPATHLENGTH+1];
   int 			tmp_linenum;

   strcpy (saved_cur_dir, curDir);
   strcpy (file_name, FullName);

   sprintf (file_name, "%s.%s", FullName, OBJ_FILE_EXT);

   if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
   if ((fp = fopen (file_name, "r")) == NULL)
   {
      if (short_name)
         sprintf (msg, "Can not open '%s'.", rest);
      else
         sprintf (msg, "Can not open '%s'.", file_name);
      Msg (msg);
      return (FALSE);
   }

   CleanUpComments ();

   strcpy (tmp_filename, scanFileName);
   tmp_linenum = scanLineNum;
   strcpy (scanFileName, (short_name ? rest : file_name));
   scanLineNum = 0;

   TieLooseEnds ();
   CleanUpDrawingWindow ();
   SetFileModified (FALSE);
   if (short_name)
      sprintf (msg, "Loading '%s' ...", rest);
   else
      sprintf (msg, "Loading '%s' ...", file_name);
   Msg (msg);
   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);

   XClearWindow (mainDisplay, drawWindow);
   somethingHighLighted = FALSE;

   numRedrawBBox = 0;
   while ((read_status = ReadObj (fp, &obj_ptr)) == TRUE)
      if (obj_ptr != NULL)
      {
         AdjForOldVersion (obj_ptr);
         AddObj (NULL, topObj, obj_ptr);
         if (PointInBBox (obj_ptr->x, obj_ptr->y, drawWinBBox) ||
               BBoxIntersect (obj_ptr->bbox, drawWinBBox))
            DrawObj (drawWindow, obj_ptr);
      }

   strcpy (scanFileName, tmp_filename);
   scanLineNum = tmp_linenum;

   if (read_status == INVALID)
   {
      sprintf (msg, "File version too large (=%1d).  Load aborted!",
            fileVersion);
      Msg (msg);
      DrawPaperBoundary ();
      RedrawGridLines ();
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);
      return (FALSE);
   }

   fclose (fp);
   SetCurDir (file_name);
   *curSymDir = '\0';
   curFileDefined = TRUE;

   if (strcmp (saved_cur_dir, curDir) != 0 && DirInSymPath ("."))
      UpdateSymInfo ();

   if (short_name)
      sprintf (msg, "Current file is '%s'.", rest);
   else
      sprintf (msg, "Current file is '%s'.", file_name);
   Msg (msg);

   RedrawTitleWindow ();
   justDupped = FALSE;
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   CleanUpCmds ();

   UpdateSubMenu (MENU_HORIALIGN);
   UpdateSubMenu (MENU_VERTALIGN);
   UpdateSubMenu (MENU_STYLE);
   UpdateSubMenu (MENU_FONT);
   UpdateSubMenu (MENU_SIZE);
   UpdateSubMenu (MENU_LINEWIDTH);
   UpdateSubMenu (MENU_LINESTYLE);
   UpdateSubMenu (MENU_LINETYPE);
   UpdateSubMenu (MENU_LINEDASH);
   UpdateSubMenu (MENU_LAYOUT);
   UpdateSubMenu (MENU_MOVEMODE);

   return (TRUE);
}

void DumpPatFill (FP, Fill, CellSize, BBox, Blanks)
   FILE		* FP;
   int		Fill, CellSize;
   struct BBRec	BBox;
   char		* Blanks;
{
   int	ltx, lty, rbx, rby;

   ltx = ((BBox.ltx % CellSize) == 0) ? BBox.ltx :
         ((BBox.ltx > 0) ? ((int)(BBox.ltx / CellSize))*CellSize :
         ((int)(BBox.ltx / CellSize)-1)*CellSize);
   lty = ((BBox.lty % CellSize) == 0) ? BBox.lty :
         ((BBox.lty > 0) ? ((int)(BBox.lty / CellSize))*CellSize :
         ((int)(BBox.lty / CellSize)-1)*CellSize);
   rbx = ((BBox.rbx % CellSize) == 0) ? BBox.rbx :
         ((BBox.rbx > 0) ? ((int)(BBox.rbx / CellSize)+1)*CellSize :
         ((int)(BBox.rbx / CellSize))*CellSize);
   rby = ((BBox.rby % CellSize) == 0) ? BBox.rby :
         ((BBox.rby > 0) ? ((int)(BBox.rby / CellSize)+1)*CellSize :
         ((int)(BBox.rby / CellSize))*CellSize);

   fprintf (FP, "%spat%1d %1d %1d %1d %1d %1d tgifpatfill\n",
         Blanks, Fill, CellSize, ltx, lty, rbx-ltx, rby-lty);
}

void DumpSymOutline (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   int  ltx, lty, rbx, rby;

   ltx = ObjPtr->obbox.ltx - QUARTER_INCH + 1;
   lty = ObjPtr->obbox.lty - QUARTER_INCH + 1;
   rbx = ObjPtr->obbox.rbx + QUARTER_INCH - 1;
   rby = ObjPtr->obbox.rby + QUARTER_INCH - 1;

   fprintf (FP, "gsave\n");
   fprintf (FP, "   0 setgray\n");
   fprintf (FP, "   [4 4] 0 setdash\n");
   fprintf (FP, "   newpath\n   %1d %1d moveto ", ltx, lty);
   fprintf (FP, "%1d %1d lineto ", rbx, lty);
   fprintf (FP, "%1d %1d lineto ", rbx, rby);
   fprintf (FP, "%1d %1d lineto\n", ltx, rby);
   fprintf (FP, "   closepath stroke\n");
   fprintf (FP, "grestore\n");
}

static
void DumpAttrs (FP, AttrPtr)
   FILE				* FP;
   register struct AttrRec	* AttrPtr;
{
   for ( ; AttrPtr != NULL; AttrPtr = AttrPtr->prev)
      if (AttrPtr->shown)
         DumpTextObj (FP, AttrPtr->obj);
}

static
void DumpAllObj (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         DumpPolyObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_BOX:
         DumpBoxObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_OVAL:
         DumpOvalObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_TEXT: DumpTextObj (FP, ObjPtr); break;
      case OBJ_POLYGON:
         DumpPolygonObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_ARC:
         DumpArcObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_RCBOX:
         DumpRCBoxObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_XBM:
         DumpXBmObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_XPM:
         DumpXPmObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr);
         break;
      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         obj_ptr = ObjPtr->detail.r->last;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
            DumpAllObj (FP, obj_ptr);
         DumpAttrs (FP, ObjPtr->lattr);
         if (ObjPtr->type == OBJ_SYM) DumpSymOutline (FP, ObjPtr);
         break;
   }
}

void DumpBBox (fp)
   FILE	* fp;
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby;
   double			llx1=0, lly1=0, urx1=0, ury1=0;

   if ((obj_ptr = topObj) == NULL)
   {
      fprintf (fp, "%%%%BoundingBox: 0 0 0 0\n");
      fprintf (stderr, "Warning:  The PostScript bounding box is empty!\n");
      Msg ("Warning:  The PostScript bounding box is empty!\n");
      return;
   }

   ltx = obj_ptr->bbox.ltx; lty = obj_ptr->bbox.lty;
   rbx = obj_ptr->bbox.rbx; rby = obj_ptr->bbox.rby;

   for (obj_ptr = topObj->next; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      if (obj_ptr->bbox.ltx < ltx) ltx = obj_ptr->bbox.ltx;
      if (obj_ptr->bbox.lty < lty) lty = obj_ptr->bbox.lty;
      if (obj_ptr->bbox.rbx > rbx) rbx = obj_ptr->bbox.rbx;
      if (obj_ptr->bbox.rby > rby) rby = obj_ptr->bbox.rby;
   }

   switch (pageStyle)
   {
      case PORTRAIT:
         llx1 = (double) (1.0*ltx*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psXOff[pageStyle]*psDotsPerInch);
         lly1 = (double) (-1.0*rby*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psYOff[pageStyle]*psDotsPerInch);
         urx1 = (double) (1.0*rbx*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psXOff[pageStyle]*psDotsPerInch);
         ury1 = (double) (-1.0*lty*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psYOff[pageStyle]*psDotsPerInch);
         break;
      case LANDSCAPE:
         llx1 = (double) (1.0*lty*psDotsPerInch/PIX_PER_INCH*printMag/100 -
               psYOff[pageStyle]*psDotsPerInch);
         lly1 = (double) (1.0*ltx*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psXOff[pageStyle]*psDotsPerInch);
         urx1 = (double) (1.0*rby*psDotsPerInch/PIX_PER_INCH*printMag/100 -
               psYOff[pageStyle]*psDotsPerInch);
         ury1 = (double) (1.0*rbx*psDotsPerInch/PIX_PER_INCH*printMag/100 +
               psXOff[pageStyle]*psDotsPerInch);
         break;
      default:
         fprintf (stderr, "Unrecognizable page style '%1d'\n", pageStyle);
         break;
   }

   llx1 += (llx1 >= 0) ? -0.001 : 0.001;
   lly1 += (lly1 >= 0) ? -0.001 : 0.001;
   urx1 += (urx1 >= 0) ? 0.001 : -0.001;
   ury1 += (ury1 >= 0) ? 0.001 : -0.001;

   fprintf (fp,"%%%%BoundingBox: %.3lf %.3lf %.3lf %.3lf\n",
         llx1, lly1, urx1, ury1);
}

void ModifyOutputFileName (FileName)
   char	* FileName;
{
   register int	i;
   int		len;
   char		s[MAXPATHLENGTH+1];

   if (*outputDir == '\0') return;

   strcpy (s, FileName);
   len = strlen (s);
   for (i = len-1; i >= 0 && s[i] != '/'; i--) ;
   if (i >= 0)
      sprintf (FileName, "%s/%s", outputDir, &s[i+1]);
   else
      sprintf (FileName, "%s/%s", outputDir, s);
}

static
void GenDump (FileName)
   char	* FileName;
{
   register struct ObjRec	* obj_ptr;
   char				cmd[MAXSTRING+1], tmp_str[MAXSTRING+1];
   char				tmp_file[MAXSTRING+1], ps_file[MAXSTRING+1];
   char				msg[MAXSTRING+1], * rest, * loc_time;
   int				i, len, short_name = FALSE;
   FILE				* fp;
   time_t			tloc;

   if (botObj == NULL) { Msg ("No object to print."); return; }

   if (!PRTGIF) Msg ("Generating print file ...");

   if (whereToPrint == XBM_FILE)
   {
      if (!curFileDefined)
      {
         if (colorDump)
            Dialog ("No current file.  Can not generate X11 pitmap output!",
                  "( <CR> or <ESC> to continue )", cmd);
         else
            Dialog ("No current file.  Can not generate X11 bitmap output!",
                  "( <CR> or <ESC> to continue )", cmd);
      }
      else
         DumpXBitmapFile ();
      return;
   }

   sprintf (tmp_file, "%sTgifXXXXXX", TMP_DIR);
   mktemp (tmp_file);
   unlink (tmp_file);

   if ((fp = fopen (tmp_file, "w")) == NULL)
   {
      if (PRTGIF)
         fprintf (stderr, "Can not create '%s', print aborted.", tmp_file);
      else
      {
         sprintf (tmp_str, "Can not create '%s', print aborted.", tmp_file);
         Msg (tmp_str);
      }
      return;
   }

   if (PRTGIF) fprintf (stderr, "Writing to '%s' ...\n", tmp_file);

   ResetGrayDetection ();

   if (usePsAdobeString)
      fprintf (fp, "%%!PS-Adobe-2.0 EPSF-1.2\n");
   else
      fprintf (fp, "%%!\n");
   DumpBBox (fp);
   if (PRTGIF)
      fprintf (fp, "%%%%Title: %s\n", FileName);
   else if (curFileDefined)
   {
      strcpy (tmp_str, curFileName);
      len = strlen (tmp_str);
      for (i = len-1; tmp_str[i] != '.'; i--) ;
      tmp_str[i] = '\0';
      fprintf (fp, "%%%%Title: %s\n", tmp_str);
   }
   else
      fprintf (fp, "%%%%Title: [Unnamed]\n");
   time (&tloc);
   loc_time = ctime (&tloc);
   loc_time[24] = '\0';
   fprintf (fp, "%%%%CreationDate: %s\n", loc_time);
   fprintf (fp, "%%%%Creator: Tgif-%s by %s (william@cs.UCLA.edu)\n",
         version_string, "William Chia-Wei Cheng");

   DumpPSMacro (fp);

   PrepareEightBitFontInfo ();
   DumpEightBitFontInfo (fp);

   fprintf (fp, "%%%%EndProlog\n");
   fprintf (fp, "%%%%Page: 1 1\n\n");
   fprintf (fp, "/tgifsavedpage save def\n\n");
   fprintf (fp, "1 setmiterlimit\n");
   fprintf (fp, "1 setlinewidth\n\n");

   if (pageStyle == LANDSCAPE) fprintf (fp, "90 rotate\n");

   fprintf (fp, "%1d %s mul %1d %s mul translate\n", psDotsPerInch,
         psXOffStr[pageStyle], psDotsPerInch, psYOffStr[pageStyle]);

   fprintf (fp, "%1d %1d div %1d mul 100 div dup neg scale\n\n",
         psDotsPerInch, PIX_PER_INCH, printMag);

   fprintf (fp, "gsave\n\n");

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      DumpAllObj (fp, obj_ptr);

   fprintf (fp, "grestore\n");
   fprintf (fp, "tgifsavedpage restore\n");
   fprintf (fp, "end\n");

   switch (whereToPrint)
   {
      case PRINTER:
      case PS_FILE: fprintf (fp, "showpage\n"); break;
      case LATEX_FIG: break;
   }
   fprintf (fp, "%%%%Trailer\n");
   fprintf (fp, "%%MatchingCreationDate: %s\n", loc_time);
   fprintf (fp, "%%%%EOF\n");
   fclose (fp);

   EndGrayDetection ();

   switch (whereToPrint)
   {
      case PRINTER:
#ifdef VMS
#define PRINT_TRAILER ""
#else
#define PRINT_TRAILER " 2>&1"
#endif
         if (PRTGIF)
         {
            if (lastFile)
               sprintf (cmd,"%s %s%s",printCommand,tmp_file,PRINT_TRAILER);
            else
               sprintf (cmd,"%s -h %s%s",printCommand,tmp_file,PRINT_TRAILER);
            fprintf (stderr, "%s\n", cmd);
         }
         else
         {
            sprintf (cmd,"%s %s%s",printCommand,tmp_file,PRINT_TRAILER);
            sprintf (msg, "Printing with '%s' command.", printCommand);
            Msg (msg);
         }
         if (!ExecuteCmd (cmd))
         {
            if (PRTGIF)
               fprintf (stderr, "Can not execute '%s', print aborted.\n", cmd);
            else
            {
               sprintf (msg, "Can not execute '%s', print aborted.", cmd);
               Msg (msg);
            }
            unlink (tmp_file);
            return;
         }
         if (PRTGIF)
            fprintf (stderr, "'%s' printed.\n\n", tmp_file);
         else
            Msg ("Print completed.");
         break;
      case LATEX_FIG:
         if (PRTGIF)
         {
            sprintf (ps_file, "%s.%s", FileName, EPSF_FILE_EXT);
            ModifyOutputFileName (ps_file);
         }
         else
         {
            if (!curFileDefined)
            {
               Dialog ("No current file.  Can not generate LaTeX output!",
                     "( <CR> or <ESC> to continue )", cmd);
               unlink (tmp_file);
               return;
            }
            sprintf (ps_file, "%s/%s", curDir, curFileName);
            len = strlen (ps_file);
            for (i = len-1; ps_file[i] != '.'; i--) ;
            sprintf (&ps_file[i], ".%s", EPSF_FILE_EXT);
            ModifyOutputFileName (ps_file);
            if (short_name = IsPrefix (bootDir, ps_file, &rest)) ++rest;
            if (short_name && *outputDir=='\0')
               sprintf (cmd, "Printing into '%s' ...", rest);
            else
               sprintf (cmd, "Printing into '%s' ...", ps_file);
            Msg (cmd);
         }
         if (!CopyAFile (tmp_file, ps_file))
         {
            if (PRTGIF)
               fprintf (stderr, "LaTeX output not generated.\n");
            else
               Msg ("LaTeX output not generated.");
            unlink (tmp_file);
            return;
         }
         if (PSFILE_MOD != 0 && chmod (ps_file, PSFILE_MOD))
         {
            if (PRTGIF)
               fprintf (stderr, "Can not chmod '%s' to 0%1o.\n", ps_file,
                     PSFILE_MOD);
            else
            {
               if (short_name && *outputDir=='\0')
                  sprintf(msg,"Can not chmod '%s' to 0%1o.",rest,PSFILE_MOD);
               else
                  sprintf(msg,"Can not chmod '%s' to 0%1o.",ps_file,PSFILE_MOD);
               Msg (msg);
            }
         }

         if (PRTGIF)
            fprintf (stderr, "LaTeX figure printed into '%s'.\n\n", ps_file);
         else
         {
            if (short_name && *outputDir=='\0')
               sprintf (msg, "LaTeX figure printed into '%s'.", rest);
            else
               sprintf (msg, "LaTeX figure printed into '%s'.", ps_file);
            Msg (msg);
         }
         break;
      case PS_FILE:
         if (PRTGIF)
         {
            sprintf (ps_file, "%s.%s", FileName, PS_FILE_EXT);
            ModifyOutputFileName (ps_file);
         }
         else
         {
            if (!curFileDefined)
            {
               Dialog ("No current file.  Can not generate PostScript output!",
                     "( <CR> or <ESC> to continue )", cmd);
               unlink (tmp_file);
               return;
            }
            sprintf (ps_file, "%s/%s", curDir, curFileName);
            len = strlen (ps_file);
            for (i = len-1; ps_file[i] != '.'; i--) ;
            sprintf (&ps_file[i], ".%s", PS_FILE_EXT);
            ModifyOutputFileName (ps_file);
            if (short_name = IsPrefix (bootDir, ps_file, &rest)) ++rest;
            if (short_name && *outputDir=='\0')
               sprintf (cmd, "Printing into '%s' ...", rest);
            else
               sprintf (cmd, "Printing into '%s' ...", ps_file);
            Msg (cmd);
         }
         if (!CopyAFile (tmp_file, ps_file))
         {
            if (PRTGIF)
               fprintf (stderr, "PostScript output not generated.\n");
            else
               Msg ("PostScript output not generated.");
            unlink (tmp_file);
            return;
         }
         if (PSFILE_MOD != 0 && chmod (ps_file, PSFILE_MOD))
         {
            if (PRTGIF)
               fprintf (stderr, "Can not chmod '%s' to 0%1o.\n", ps_file,
                     PSFILE_MOD);
            else
            {
               if (short_name && *outputDir=='\0')
                  sprintf(msg,"Can not chmod '%s' to 0%1o.",rest,PSFILE_MOD);
               else
                  sprintf(msg,"Can not chmod '%s' to 0%1o.",ps_file,PSFILE_MOD);
               Msg (msg);
            }
         }
         if (PRTGIF)
            fprintf (stderr, "PostScript file printed into '%s'.\n\n", ps_file);
         else
         {
            if (short_name && *outputDir=='\0')
               sprintf (msg, "PostScript file printed into '%s'.", rest);
            else
               sprintf (msg, "PostScript file printed into '%s'.", ps_file);
            Msg (msg);
         }
         break;
   }
#ifdef KEEP_WHEN_PRINT
   if (whereToPrint != PRINTER) unlink (tmp_file);
#else
   unlink (tmp_file);
#endif
}

void Dump (FileName)
   char	* FileName;
{
   int	len, obj_ext_len, sym_ext_len;
   char	obj_ext_str[MAXSTRING+1], sym_ext_str[MAXSTRING+1];

   sprintf (obj_ext_str, ".%s", OBJ_FILE_EXT);
   obj_ext_len = strlen (obj_ext_str);
   sprintf (sym_ext_str, ".%s", SYM_FILE_EXT);
   sym_ext_len = strlen (sym_ext_str);

   if (PRTGIF)
   {
      len = strlen (FileName);
      if ((len >= obj_ext_len &&
            strcmp (&FileName[len-obj_ext_len], obj_ext_str) == 0) ||
            (len >= sym_ext_len &&
            strcmp (&FileName[len-sym_ext_len], sym_ext_str) == 0))
         FileName[len-obj_ext_len] = '\0';
   }
   else
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
   }
   GenDump (FileName);
   if (!PRTGIF)
   {
      SetDefaultCursor (mainWindow);
      ShowCursor ();
   }
}

void PrintWithCommand (FileName)
   char	* FileName;
{
   if (whereToPrint != PRINTER)
   {
      Msg ("PrintWithCmd only works when output device is the printer.");
      return;
   }
   Dialog ("Please Enter Print Command Name:",
         "( <CR>: accept, <ESC>: cancel )", printCommand);
   if (*printCommand == '\0') return;
   GenDump (FileName);
}

void SetPrintReduction ()
{
   int	value;
   char	buf[MAXSTRING+1], msg[MAXSTRING+1];

   Dialog ("Please specify percent reduction (<100) or enlargement (>100):",
         "( <CR>: accept, <ESC>: cancel )", buf);
   if (*buf == '\0') return;

   value = atoi (buf);
   if (value <= 0)
   {
      sprintf (msg, "Invalid reduction '%s'.", buf);
      Msg (msg);
      return;
   }
   printMag = value;
   if (UpdPageStyle (pageStyle))
   {
      UpdDrawWinBBox ();
      AdjSplineVs ();
      ClearAndRedrawDrawWindow ();
   }
   RedrawRulers ();
   RedrawTitleWindow ();
   SetFileModified (TRUE);
   if (printMag <= 100)
      sprintf (msg, "New reduction is %1d%%.", printMag);
   else
      sprintf (msg, "New enlargement is %1d%%.", printMag);
   Msg (msg);
}

void NewProc ()
{
   while (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before clear? [ync](y)",
            CONFIRM_YES))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: TieLooseEnds (); SetFileModified (FALSE); break;
         case CONFIRM_CANCEL: return;
      }
   }
   CleanUpComments ();
   CleanUpCmds ();
   CleanUpDrawingWindow ();
   ClearFileInfo ();
   ClearAndRedrawDrawWindow ();
   Msg ("Editing no file.");
   objId = 0;
   RedrawTitleWindow ();
}

void OpenProc ()
{
   char	file_name[MAXPATHLENGTH+1];
   int	do_not_save = FALSE;

   while (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before open? [ync](y)",
            CONFIRM_YES))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: do_not_save = TRUE; SetFileModified (FALSE); break;
         case CONFIRM_CANCEL: return;
      }
   }
   if (SelectFileName ("Please select a file to OPEN ...",file_name) != INVALID)
   {
      char	ext_str[MAXSTRING+1];
      int	ext_len, len=strlen(file_name);

      sprintf (ext_str, ".%s", OBJ_FILE_EXT);
      ext_len = strlen (ext_str);
      if (len > ext_len && strcmp(ext_str, &file_name[len-ext_len]) == 0)
         file_name[len-ext_len] = '\0';

      if (!LoadFile (file_name) && do_not_save)
         SetFileModified (TRUE);
   }
   else if (do_not_save)
      SetFileModified (TRUE);

   SetDefaultCursor (mainWindow);
   ShowCursor ();
}

char * fileMenuStr[] =
      { "New            ^n",
        "Open           ^o",
        "Save           ^s",
        "SaveNew       ^#s",
        "Import         #p",
        "ImportXBitmap ^#.",
        "ImportXPixmap ^#,",
        "ImportEPSFile  #(",
        "ChangeDomain   ^c",
        "Print          ^p",
        "PrintWithCmd   ^-",
        "Solve          #s",
        "Simulate       #y",
        "Probe          #b",
        "Animate        ^z",
        "Escape         #x",
        "SaveSelectedAs #~",
        "SaveSymInLibrary ",
        "Quit           ^q"
      };

int QuitProc ()
{
   int	do_not_save = FALSE;

   while (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before quit? [ync](y)",
            CONFIRM_YES))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: do_not_save = TRUE; SetFileModified (FALSE); break;
         case CONFIRM_CANCEL: return (INVALID);
      }
   }
   if (AncesterModified ())
   {
      switch (YesNoCancel ("Ancester file modified, still quitting? [ync](y)",
            CONFIRM_YES))
      {
         case CONFIRM_YES: return (FILE_QUIT);
         case CONFIRM_NO:
            if (do_not_save) SetFileModified (TRUE);
            return (INVALID);
         case CONFIRM_CANCEL:
            if (do_not_save) SetFileModified (TRUE);
            return (INVALID);
      }
   }
   return (FILE_QUIT);
}

int SolveProc ()
{
   if (!saveTmpOnReturn) return (FILE_SOLVE);

   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_SOLVE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
   return (INVALID);
}

int SimulateProc ()
{
   if (!saveTmpOnReturn) return (FILE_SIMULATE);

   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_SIMULATE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
   return (INVALID);
}

int ProbeProc ()
{
   if (!saveTmpOnReturn) return (FILE_PROBE);

   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_PROBE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
   return (INVALID);
}

int AnimateProc ()
{
   if (!saveTmpOnReturn) return (FILE_ANIMATE);

   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_ANIMATE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
   return (INVALID);
}

int EscapeProc ()
{
   return (FILE_ESCAPE);
}

int FileSubMenu (index)
   int	index;
{
   switch (index)
   {
      case FILE_NEW: NewProc (); break;
      case FILE_OPEN: OpenProc (); break;
      case FILE_SAVE: SaveFile (); break;
      case FILE_SAVENEW: SaveNewFile (FALSE); break;
      case FILE_IMPORT: ImportFile (); break;
      case FILE_IMPORTXBM: ImportXBitmapFile (); break;
      case FILE_IMPORTXPM: ImportXPixmapFile (); break;
      case FILE_IMPORTEPS: ImportEPSFile (); break;
      case FILE_DOMAIN: ChangeDomain (); break;
      case FILE_DUMP: Dump (""); break;
      case FILE_USR_DUMP: PrintWithCommand (""); break;
      case FILE_SOLVE: return (SolveProc ());
      case FILE_SIMULATE: return (SimulateProc ());
      case FILE_PROBE: return (ProbeProc ());
      case FILE_ANIMATE: return (AnimateProc ());
      case FILE_ESCAPE: return (EscapeProc ());
      case FILE_SAVESELAS: SaveNewFile (TRUE); break;
      case FILE_SAVESYMINLIB: SaveSymInLibrary (); break;
      case FILE_QUIT: return (QuitProc ());
   }
   return (INVALID);
}

int FileMenu (X, Y)
   int	X, Y;
{
   register int index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (FILEMENUENTRIES, &fore_colors, &valid, &init_rv);
   activeMenu = MENU_FILE;
   index = TextMenuLoop (X, Y, fileMenuStr, FILEMENUENTRIES, fore_colors,
         valid, init_rv, SINGLECOLOR);

   if (index != INVALID) return (FileSubMenu (index));
   return (INVALID);
}

void CleanUpFiles ()
{
   CleanUpComments ();
   ClearFileInfo ();
   fileModified = FALSE;
}
